<?php
/**
 * Plugin Name: TTB Woo Timeslot Bookings
 * Description: Simple timeslot booking for WooCommerce products with configurable weekday slots, shared capacity, calendar UI and admin tools.
 * Version: 1.4.0
 * Author: Dale / Five12
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class TTB_Woo_Timeslot_Bookings {

    const TABLE_NAME   = 'ttb_product_timeslots'; // will be prefixed
    const OPTION_SLOTS = 'ttb_timeslot_definitions';

    // How many days ahead to allow bookings (120 ~ 4 months)
    const DAYS_AHEAD = 120;

    // Default slots (used only as fallback / first install)
    const SLOT_1_START = '10:00';
    const SLOT_1_END   = '11:30';
    const SLOT_2_START = '11:30';
    const SLOT_2_END   = '13:00';

    // Dates where no slots should be offered at all (hard-coded bank holidays etc)
    // Format: 'YYYY-MM-DD'
    const BLACKOUT_DATES = array(
        // '2025-12-25',
        // '2025-12-26',
    );

    public function __construct() {
        register_activation_hook( __FILE__, array( $this, 'activate' ) );

        // Frontend assets
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );

        // Frontend hooks
        add_action( 'woocommerce_before_add_to_cart_button', array( $this, 'render_timeslot_selector' ) );
        add_filter( 'woocommerce_add_to_cart_validation', array( $this, 'validate_timeslot_on_add_to_cart' ), 10, 3 );
        add_filter( 'woocommerce_add_cart_item_data', array( $this, 'add_timeslot_to_cart_item' ), 10, 3 );
        add_filter( 'woocommerce_get_item_data', array( $this, 'display_timeslot_in_cart' ), 10, 2 );

        // Save into order items
        add_action( 'woocommerce_checkout_create_order_line_item', array( $this, 'add_timeslot_to_order_items' ), 10, 4 );

        // Mark slot as booked when order is paid
        add_action( 'woocommerce_order_status_processing', array( $this, 'lock_timeslots_for_order' ) );
        add_action( 'woocommerce_order_status_completed', array( $this, 'lock_timeslots_for_order' ) );

        // Admin menu + assets
        add_action( 'admin_menu', array( $this, 'register_admin_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_assets' ) );
    }

    /**
     * Plugin activation hook. Creates custom table and sets default slots.
     */
    public function activate() {
        global $wpdb;

        $table_name      = $this->get_table_name();
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            product_id BIGINT(20) UNSIGNED NOT NULL,
            slot_date DATE NOT NULL,
            slot_start TIME NOT NULL,
            slot_end TIME NOT NULL,
            order_id BIGINT(20) UNSIGNED DEFAULT NULL,
            status VARCHAR(20) NOT NULL DEFAULT 'booked',
            PRIMARY KEY (id),
            UNIQUE KEY slot_unique (slot_date, slot_start, status)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );

        // Initialise slots option if empty
        $slots = get_option( self::OPTION_SLOTS );
        if ( ! is_array( $slots ) || empty( $slots ) ) {
            $default_slots = array(
                array(
                    'start' => self::SLOT_1_START,
                    'end'   => self::SLOT_1_END,
                ),
                array(
                    'start' => self::SLOT_2_START,
                    'end'   => self::SLOT_2_END,
                ),
            );
            update_option( self::OPTION_SLOTS, $default_slots );
        }
    }

    protected function get_table_name() {
        global $wpdb;
        return $wpdb->prefix . self::TABLE_NAME;
    }

    /**
     * Get configured time slots from options.
     * Format: array( array( 'start' => 'HH:MM', 'end' => 'HH:MM' ), ... )
     */
    protected function get_defined_slots() {
        $slots = get_option( self::OPTION_SLOTS );

        if ( ! is_array( $slots ) ) {
            $slots = array();
        }

        // Basic sanitise / normalise
        $clean = array();
        foreach ( $slots as $slot ) {
            if ( empty( $slot['start'] ) || empty( $slot['end'] ) ) {
                continue;
            }
            $start = $this->normalise_time( $slot['start'] );
            $end   = $this->normalise_time( $slot['end'] );
            if ( ! $start || ! $end ) {
                continue;
            }
            if ( $start >= $end ) {
                continue;
            }

            $clean[] = array(
                'start' => $start,
                'end'   => $end,
            );
        }

        // If still empty, use defaults as a safety net
        if ( empty( $clean ) ) {
            $clean = array(
                array(
                    'start' => self::SLOT_1_START,
                    'end'   => self::SLOT_1_END,
                ),
                array(
                    'start' => self::SLOT_2_START,
                    'end'   => self::SLOT_2_END,
                ),
            );
        }

        // Sort by start time
        usort(
            $clean,
            function( $a, $b ) {
                return strcmp( $a['start'], $b['start'] );
            }
        );

        return $clean;
    }

    /**
     * Normalise a time string to HH:MM, returns false if invalid.
     */
    protected function normalise_time( $time ) {
        $time = trim( (string) $time );
        if ( $time === '' ) {
            return false;
        }

        // Accept H:MM or HH:MM
        if ( ! preg_match( '/^(\d{1,2}):(\d{2})$/', $time, $m ) ) {
            return false;
        }

        $h = (int) $m[1];
        $i = (int) $m[2];

        if ( $h < 0 || $h > 23 || $i < 0 || $i > 59 ) {
            return false;
        }

        return sprintf( '%02d:%02d', $h, $i );
    }

    /**
     * Frontend assets (datepicker).
     */
    public function enqueue_assets() {
        if ( ! is_product() ) {
            return;
        }

        wp_enqueue_script( 'jquery-ui-datepicker' );
        wp_enqueue_style(
            'jquery-ui-theme',
            'https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css',
            array(),
            '1.13.2'
        );
    }

    /**
     * Admin assets (datepicker).
     */
    public function admin_enqueue_assets( $hook ) {
        if ( $hook !== 'woocommerce_page_ttb-timeslot-bookings' ) {
            return;
        }

        wp_enqueue_script( 'jquery-ui-datepicker' );
        wp_enqueue_style(
            'jquery-ui-theme-admin',
            'https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css',
            array(),
            '1.13.2'
        );
    }

    /**
     * Admin menu: WooCommerce -> Timeslot Bookings.
     */
    public function register_admin_menu() {
        add_submenu_page(
            'woocommerce',
            'Timeslot Bookings',
            'Timeslot Bookings',
            'manage_woocommerce',
            'ttb-timeslot-bookings',
            array( $this, 'render_admin_page' )
        );
    }

    /**
     * Admin page controller.
     */
    public function render_admin_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        $this->handle_admin_actions();

        ?>
        <div class="wrap">
            <h1>Timeslot Bookings</h1>

            <h2 class="nav-tab-wrapper">
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=ttb-timeslot-bookings&tab=bookings' ) ); ?>" class="nav-tab <?php echo $this->get_active_tab() === 'bookings' ? 'nav-tab-active' : ''; ?>">Current bookings</a>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=ttb-timeslot-bookings&tab=manual' ) ); ?>" class="nav-tab <?php echo $this->get_active_tab() === 'manual' ? 'nav-tab-active' : ''; ?>">Add manual booking</a>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=ttb-timeslot-bookings&tab=block' ) ); ?>" class="nav-tab <?php echo $this->get_active_tab() === 'block' ? 'nav-tab-active' : ''; ?>">Block dates / times</a>
                <a href="<?php echo esc_url( admin_url( 'admin.php?page=ttb-timeslot-bookings&tab=slots' ) ); ?>" class="nav-tab <?php echo $this->get_active_tab() === 'slots' ? 'nav-tab-active' : ''; ?>">Time slot settings</a>
            </h2>
        <?php

        switch ( $this->get_active_tab() ) {
            case 'manual':
                $this->render_manual_booking_form();
                break;
            case 'block':
                $this->render_block_form();
                break;
            case 'slots':
                $this->render_slots_settings_form();
                break;
            case 'bookings':
            default:
                $this->render_bookings_table();
                break;
        }

        echo '</div>';
    }

    protected function get_active_tab() {
        return isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'bookings';
    }

    /**
     * Handle admin form actions (manual booking, block, cancel, slot settings).
     */
    protected function handle_admin_actions() {
        global $wpdb;
        $table = $this->get_table_name();

        // Slot settings form
        if (
            isset( $_POST['ttb_slots_nonce'] )
            && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['ttb_slots_nonce'] ) ), 'ttb_slots_form' )
        ) {
            $starts = isset( $_POST['ttb_slots_start'] ) ? (array) $_POST['ttb_slots_start'] : array();
            $ends   = isset( $_POST['ttb_slots_end'] ) ? (array) $_POST['ttb_slots_end'] : array();

            $new_slots = array();

            $max = max( count( $starts ), count( $ends ) );

            for ( $i = 0; $i < $max; $i++ ) {
                $start_raw = isset( $starts[ $i ] ) ? sanitize_text_field( wp_unslash( $starts[ $i ] ) ) : '';
                $end_raw   = isset( $ends[ $i ] ) ? sanitize_text_field( wp_unslash( $ends[ $i ] ) ) : '';

                if ( $start_raw === '' && $end_raw === '' ) {
                    continue;
                }

                $start = $this->normalise_time( $start_raw );
                $end   = $this->normalise_time( $end_raw );

                if ( ! $start || ! $end ) {
                    continue;
                }
                if ( $start >= $end ) {
                    continue;
                }

                $new_slots[] = array(
                    'start' => $start,
                    'end'   => $end,
                );
            }

            // Sort by start time
            usort(
                $new_slots,
                function( $a, $b ) {
                    return strcmp( $a['start'], $b['start'] );
                }
            );

            update_option( self::OPTION_SLOTS, $new_slots );
            echo '<div class="notice notice-success is-dismissible"><p>Time slots updated.</p></div>';
        }

        // Manual booking form
        if (
            isset( $_POST['ttb_manual_booking_nonce'] )
            && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['ttb_manual_booking_nonce'] ) ), 'ttb_manual_booking' )
        ) {
            $product_id = isset( $_POST['ttb_manual_product'] ) ? (int) $_POST['ttb_manual_product'] : 0;
            $date       = isset( $_POST['ttb_manual_date'] ) ? sanitize_text_field( wp_unslash( $_POST['ttb_manual_date'] ) ) : '';
            $slot       = isset( $_POST['ttb_manual_time'] ) ? sanitize_text_field( wp_unslash( $_POST['ttb_manual_time'] ) ) : '';

            if ( $product_id && $date && $slot ) {
                $parts = explode( '|', $slot );
                if ( count( $parts ) === 2 ) {
                    $start = substr( $parts[0], 0, 5 );
                    $end   = substr( $parts[1], 0, 5 );

                    if ( $this->is_slot_available( $date, $start ) ) {
                        $wpdb->insert(
                            $table,
                            array(
                                'product_id' => $product_id,
                                'slot_date'  => $date,
                                'slot_start' => $start,
                                'slot_end'   => $end,
                                'order_id'   => 0,
                                'status'     => 'booked',
                            ),
                            array( '%d', '%s', '%s', '%s', '%d', '%s' )
                        );
                        echo '<div class="notice notice-success is-dismissible"><p>Manual booking added.</p></div>';
                    } else {
                        echo '<div class="notice notice-error is-dismissible"><p>That slot is already booked or blocked.</p></div>';
                    }
                }
            } else {
                echo '<div class="notice notice-error is-dismissible"><p>Please select product, date and time.</p></div>';
            }
        }

        // Block slot / day form
        if (
            isset( $_POST['ttb_block_nonce'] )
            && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['ttb_block_nonce'] ) ), 'ttb_block_form' )
        ) {
            $date       = isset( $_POST['ttb_block_date'] ) ? sanitize_text_field( wp_unslash( $_POST['ttb_block_date'] ) ) : '';
            $block_type = isset( $_POST['ttb_block_type'] ) ? sanitize_key( wp_unslash( $_POST['ttb_block_type'] ) ) : 'day';
            $block_slot = isset( $_POST['ttb_block_time'] ) ? sanitize_text_field( wp_unslash( $_POST['ttb_block_time'] ) ) : '';

            if ( $date ) {
                $slots_config = $this->get_defined_slots();
                $slots_to_block = array();

                if ( $block_type === 'day' ) {
                    // Block all configured slots
                    foreach ( $slots_config as $slot ) {
                        $slots_to_block[] = array( $slot['start'], $slot['end'] );
                    }
                } elseif ( $block_type === 'slot' && $block_slot ) {
                    $parts = explode( '|', $block_slot );
                    if ( count( $parts ) === 2 ) {
                        $slots_to_block[] = array(
                            substr( $parts[0], 0, 5 ),
                            substr( $parts[1], 0, 5 ),
                        );
                    }
                }

                foreach ( $slots_to_block as $slot ) {
                    list( $start, $end ) = $slot;
                    $start_short = substr( $start, 0, 5 );
                    $end_short   = substr( $end, 0, 5 );

                    if ( $this->is_slot_available( $date, $start_short ) ) {
                        $wpdb->insert(
                            $table,
                            array(
                                'product_id' => 0,
                                'slot_date'  => $date,
                                'slot_start' => $start_short,
                                'slot_end'   => $end_short,
                                'order_id'   => 0,
                                'status'     => 'blocked',
                            ),
                            array( '%d', '%s', '%s', '%s', '%d', '%s' )
                        );
                    }
                }

                echo '<div class="notice notice-success is-dismissible"><p>Block saved.</p></div>';
            } else {
                echo '<div class="notice notice-error is-dismissible"><p>Please select a date to block.</p></div>';
            }
        }

        // Cancel booking / unblock
        if (
            isset( $_GET['ttb_cancel'] )
            && isset( $_GET['_ttb_nonce'] )
            && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_ttb_nonce'] ) ), 'ttb_cancel_booking' )
        ) {
            $id = (int) $_GET['ttb_cancel'];

            if ( $id > 0 ) {
                $wpdb->update(
                    $table,
                    array( 'status' => 'cancelled' ),
                    array( 'id' => $id ),
                    array( '%s' ),
                    array( '%d' )
                );
                echo '<div class="notice notice-success is-dismissible"><p>Booking/block cancelled. Slot is now available again.</p></div>';
            }
        }
    }

    /**
     * Admin: manual booking form.
     */
    protected function render_manual_booking_form() {
        $products = get_posts(
            array(
                'post_type'      => 'product',
                'posts_per_page' => -1,
                'post_status'    => 'publish',
                'orderby'        => 'title',
                'order'          => 'ASC',
            )
        );

        $slots = $this->get_defined_slots();
        ?>
        <h2>Add manual booking</h2>
        <p>Use this form to assign a timeslot to a product without going through the front-end checkout.</p>

        <form method="post">
            <?php wp_nonce_field( 'ttb_manual_booking', 'ttb_manual_booking_nonce' ); ?>

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="ttb_manual_product">Product</label></th>
                    <td>
                        <select name="ttb_manual_product" id="ttb_manual_product">
                            <option value="">Select a product</option>
                            <?php foreach ( $products as $p ) : ?>
                                <option value="<?php echo (int) $p->ID; ?>"><?php echo esc_html( $p->post_title ); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="ttb_manual_date">Date</label></th>
                    <td>
                        <input type="text" name="ttb_manual_date" id="ttb_manual_date" class="ttb-date" autocomplete="off" />
                        <p class="description">Weekdays only.</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="ttb_manual_time">Timeslot</label></th>
                    <td>
                        <select name="ttb_manual_time" id="ttb_manual_time">
                            <option value="">Select a time</option>
                            <?php foreach ( $slots as $slot ) : ?>
                                <?php
                                $val   = $slot['start'] . '|' . $slot['end'];
                                $label = $slot['start'] . ' - ' . $slot['end'];
                                ?>
                                <option value="<?php echo esc_attr( $val ); ?>"><?php echo esc_html( $label ); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                </tr>
            </table>

            <?php submit_button( 'Add booking' ); ?>
        </form>

        <script>
        jQuery(function($) {
            $('#ttb_manual_date').datepicker({
                dateFormat: 'yy-mm-dd',
                minDate: 0,
                beforeShowDay: function(date) {
                    var day = date.getDay(); // 0 = Sun, 6 = Sat
                    if (day === 0 || day === 6) {
                        return [false, '', 'Weekends not bookable'];
                    }
                    return [true, '', ''];
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Admin: block dates / times form.
     */
    protected function render_block_form() {
        $slots = $this->get_defined_slots();
        ?>
        <h2>Block dates / times</h2>
        <p>Use this form to block out timeslots (for example holidays or days off). Blocks are global, not tied to a specific product.</p>

        <form method="post">
            <?php wp_nonce_field( 'ttb_block_form', 'ttb_block_nonce' ); ?>

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="ttb_block_date">Date to block</label></th>
                    <td>
                        <input type="text" name="ttb_block_date" id="ttb_block_date" class="ttb-date" autocomplete="off" />
                        <p class="description">Weekdays only.</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><label for="ttb_block_type">Block type</label></th>
                    <td>
                        <select name="ttb_block_type" id="ttb_block_type">
                            <option value="day">Block whole day (all defined slots)</option>
                            <option value="slot">Block a single slot</option>
                        </select>
                    </td>
                </tr>
                <tr class="ttb-block-slot-row">
                    <th scope="row"><label for="ttb_block_time">Slot to block</label></th>
                    <td>
                        <select name="ttb_block_time" id="ttb_block_time">
                            <option value="">Select a slot</option>
                            <?php foreach ( $slots as $slot ) : ?>
                                <?php
                                $val   = $slot['start'] . '|' . $slot['end'];
                                $label = $slot['start'] . ' - ' . $slot['end'];
                                ?>
                                <option value="<?php echo esc_attr( $val ); ?>"><?php echo esc_html( $label ); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description">Only used if "Block a single slot" is selected.</p>
                    </td>
                </tr>
            </table>

            <?php submit_button( 'Block selected time(s)' ); ?>
        </form>

        <script>
        jQuery(function($) {
            $('#ttb_block_date').datepicker({
                dateFormat: 'yy-mm-dd',
                minDate: 0,
                beforeShowDay: function(date) {
                    var day = date.getDay(); // 0 = Sun, 6 = Sat
                    if (day === 0 || day === 6) {
                        return [false, '', 'Weekends not bookable'];
                    }
                    return [true, '', ''];
                }
            });

            function toggleSlotRow() {
                var type = $('#ttb_block_type').val();
                if (type === 'slot') {
                    $('.ttb-block-slot-row').show();
                } else {
                    $('.ttb-block-slot-row').hide();
                }
            }

            $('#ttb_block_type').on('change', toggleSlotRow);
            toggleSlotRow();
        });
        </script>
        <?php
    }

    /**
     * Admin: time slot settings form.
     */
    protected function render_slots_settings_form() {
        $slots = $this->get_defined_slots();
        ?>
        <h2>Time slot settings</h2>
        <p>Define the time slots available Monday to Friday. These slots are used on the product calendar, manual booking form and block form.</p>

        <form method="post">
            <?php wp_nonce_field( 'ttb_slots_form', 'ttb_slots_nonce' ); ?>

            <table class="widefat fixed" id="ttb-slots-table">
                <thead>
                    <tr>
                        <th style="width: 150px;">Start time (HH:MM)</th>
                        <th style="width: 150px;">End time (HH:MM)</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody id="ttb-slots-table-body">
                    <?php if ( ! empty( $slots ) ) : ?>
                        <?php foreach ( $slots as $slot ) : ?>
                            <tr>
                                <td><input type="text" name="ttb_slots_start[]" value="<?php echo esc_attr( $slot['start'] ); ?>" /></td>
                                <td><input type="text" name="ttb_slots_end[]" value="<?php echo esc_attr( $slot['end'] ); ?>" /></td>
                                <td><a href="#" class="button ttb-remove-slot">Remove</a></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    <!-- Empty row template will be added via JS -->
                </tbody>
            </table>

            <p>
                <a href="#" class="button" id="ttb-add-slot">Add slot</a>
            </p>

            <?php submit_button( 'Save slots' ); ?>
        </form>

        <script>
        jQuery(function($) {
            var $tbody = $('#ttb-slots-table-body');

            function addSlotRow(start, end) {
                start = start || '';
                end   = end || '';
                var row = '<tr>' +
                    '<td><input type="text" name="ttb_slots_start[]" value="' + start + '" placeholder="10:00" /></td>' +
                    '<td><input type="text" name="ttb_slots_end[]" value="' + end + '" placeholder="11:30" /></td>' +
                    '<td><a href="#" class="button ttb-remove-slot">Remove</a></td>' +
                '</tr>';
                $tbody.append(row);
            }

            $('#ttb-add-slot').on('click', function(e) {
                e.preventDefault();
                addSlotRow('', '');
            });

            $tbody.on('click', '.ttb-remove-slot', function(e) {
                e.preventDefault();
                $(this).closest('tr').remove();
            });

            // If no slots exist for some reason, add one blank row
            if ($tbody.find('tr').length === 0) {
                addSlotRow('', '');
            }
        });
        </script>
        <?php
    }

    /**
     * Admin: bookings table.
     */
    protected function render_bookings_table() {
        global $wpdb;
        $table = $this->get_table_name();

        $today = current_time( 'Y-m-d' );

        $rows = $wpdb->get_results(
            $wpdb->prepare(
                "
                SELECT *
                FROM {$table}
                WHERE slot_date >= %s
                  AND status IN ('booked','blocked')
                ORDER BY slot_date ASC, slot_start ASC
                ",
                $today
            )
        );

        ?>
        <h2>Current bookings and blocks</h2>
        <p>These are upcoming slots that are currently booked or blocked. Cancel a row to free the slot again.</p>

        <table class="widefat fixed striped">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Date</th>
                    <th>Time</th>
                    <th>Type</th>
                    <th>Product</th>
                    <th>Order</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
            <?php if ( empty( $rows ) ) : ?>
                <tr><td colspan="7">No upcoming bookings or blocks.</td></tr>
            <?php else : ?>
                <?php foreach ( $rows as $row ) : ?>
                    <?php
                    $date_disp  = date_i18n( 'D j M Y', strtotime( $row->slot_date ) );
                    $time_label = substr( $row->slot_start, 0, 5 ) . ' - ' . substr( $row->slot_end, 0, 5 );
                    $type_label = $row->status === 'blocked' ? 'Blocked' : 'Booked';

                    $product_title = '';
                    if ( $row->product_id ) {
                        $p = get_post( $row->product_id );
                        if ( $p ) {
                            $product_title = $p->post_title;
                        }
                    } else {
                        $product_title = $row->status === 'blocked' ? '(Global block)' : '';
                    }

                    $order_link = '';
                    if ( $row->order_id ) {
                        $order_link = sprintf(
                            '<a href="%s">#%d</a>',
                            esc_url( get_edit_post_link( $row->order_id ) ),
                            (int) $row->order_id
                        );
                    }

                    $cancel_url = wp_nonce_url(
                        add_query_arg(
                            array(
                                'page'       => 'ttb-timeslot-bookings',
                                'tab'        => 'bookings',
                                'ttb_cancel' => (int) $row->id,
                            ),
                            admin_url( 'admin.php' )
                        ),
                        'ttb_cancel_booking',
                        '_ttb_nonce'
                    );
                    ?>
                    <tr>
                        <td><?php echo (int) $row->id; ?></td>
                        <td><?php echo esc_html( $date_disp ); ?></td>
                        <td><?php echo esc_html( $time_label ); ?></td>
                        <td><?php echo esc_html( $type_label ); ?></td>
                        <td><?php echo esc_html( $product_title ); ?></td>
                        <td><?php echo wp_kses_post( $order_link ); ?></td>
                        <td>
                            <a href="<?php echo esc_url( $cancel_url ); ?>" class="button button-small">Cancel / unblock</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
        </table>
        <?php
    }

    /**
     * Frontend: render timeslot selector (calendar + options).
     */
    public function render_timeslot_selector() {
        global $product;

        if ( ! $product instanceof WC_Product ) {
            return;
        }

        // Only for simple products by default, tweak if needed
        if ( $product->get_type() !== 'simple' ) {
            return;
        }

        // Build calendar data (available dates + slots per date, shared across products)
        $calendar_data = $this->get_calendar_data();

        $enabled_dates = $calendar_data['enabled_dates'];
        $slots_map     = $calendar_data['slots_map'];

        if ( empty( $enabled_dates ) ) {
            echo '<p class="ttb-timeslots__notice">No timeslots available at the moment.</p>';
            return;
        }

        $enabled_dates_json = esc_attr( wp_json_encode( $enabled_dates ) );
        $slots_map_json     = esc_attr( wp_json_encode( $slots_map ) );
        $max_days           = (int) self::DAYS_AHEAD - 1;
        ?>
        <div class="ttb-timeslots"
             data-enabled-dates="<?php echo $enabled_dates_json; ?>"
             data-slots-map="<?php echo $slots_map_json; ?>"
             data-max-days="<?php echo $max_days; ?>">

            <h4 class="ttb-timeslots__label"><?php esc_html_e( 'Select a date and time', 'ttb-woo-timeslot-bookings' ); ?></h4>

            <div class="ttb-timeslots__row">
                <div class="ttb-timeslots__col">
                    <label for="ttb-timeslots-date" class="ttb-timeslots__field-label">
                        <?php esc_html_e( 'Choose a date', 'ttb-woo-timeslot-bookings' ); ?>
                    </label>
                    <input
                        type="text"
                        id="ttb-timeslots-date"
                        class="ttb-timeslots__date"
                        placeholder="<?php esc_attr_e( 'Click to pick a date', 'ttb-woo-timeslot-bookings' ); ?>"
                        autocomplete="off"
                    />
                </div>
                <div class="ttb-timeslots__col">
                    <label class="ttb-timeslots__field-label">
                        <?php esc_html_e( 'Available time slots', 'ttb-woo-timeslot-bookings' ); ?>
                    </label>
                    <div class="ttb-timeslots__times">
                        <p class="ttb-timeslots__hint">
                            <?php esc_html_e( 'Choose a date to see available times.', 'ttb-woo-timeslot-bookings' ); ?>
                        </p>
                    </div>
                </div>
            </div>

            <input type="hidden" name="ttb_timeslot" class="ttb-timeslots__value" value="" />
        </div>

        <style>
            .ttb-timeslots {
                margin: 1.25em 0;
                padding: 1em;
                border: 1px solid #e2e2e2;
                border-radius: 6px;
                background: #fafafa;
            }
            .ttb-timeslots__label {
                margin: 0 0 0.75em;
                font-weight: 600;
                font-size: 15px;
            }
            .ttb-timeslots__row {
                display: flex;
                flex-wrap: wrap;
                gap: 1em;
            }
            .ttb-timeslots__col {
                flex: 1 1 200px;
                min-width: 0;
            }
            .ttb-timeslots__field-label {
                display: block;
                font-size: 13px;
                margin-bottom: 0.25em;
                color: #444;
            }
            .ttb-timeslots__date {
                width: 100%;
                max-width: 240px;
                padding: 0.5em 0.75em;
                border-radius: 4px;
                border: 1px solid #ccc;
                font-size: 14px;
                background-color: #fff;
                cursor: pointer;
            }
            .ttb-timeslots__times {
                font-size: 14px;
            }
            .ttb-timeslots__hint {
                margin: 0.4em 0 0;
                color: #777;
            }
            .ttb-timeslots__options {
                display: flex;
                flex-direction: column;
                gap: 0.35em;
                margin-top: 0.35em;
            }
            .ttb-timeslots__option {
                display: inline-flex;
                align-items: center;
                gap: 0.35em;
                padding: 0.35em 0.5em;
                border-radius: 4px;
                border: 1px solid #e0e0e0;
                background: #fff;
                cursor: pointer;
            }
            .ttb-timeslots__option input[type="radio"] {
                margin: 0;
            }
            .ttb-timeslots__notice {
                font-size: 14px;
                color: #777;
            }

            @media (max-width: 600px) {
                .ttb-timeslots__row {
                    flex-direction: column;
                }
            }
        </style>

        <script>
            jQuery(function($) {
                var $wrap = $('.ttb-timeslots').last(); // only one on the product page
                if (!$wrap.length) {
                    return;
                }

                var enabledDates = [];
                var slotsMap = {};
                try {
                    enabledDates = JSON.parse($wrap.attr('data-enabled-dates') || '[]');
                    slotsMap = JSON.parse($wrap.attr('data-slots-map') || '{}');
                } catch (e) {
                    console && console.error('TTB Timeslots JSON parse error', e);
                }

                var enabledSet = {};
                enabledDates.forEach(function(d) { enabledSet[d] = true; });

                var maxDays = parseInt($wrap.attr('data-max-days') || '120', 10);

                var $dateInput = $wrap.find('.ttb-timeslots__date');
                var $timesWrap = $wrap.find('.ttb-timeslots__times');
                var $valueInput = $wrap.find('.ttb-timeslots__value');

                if (!$dateInput.length) {
                    return;
                }

                $dateInput.datepicker({
                    dateFormat: 'yy-mm-dd',
                    minDate: 0,
                    maxDate: maxDays,
                    beforeShowDay: function(date) {
                        var y = date.getFullYear();
                        var m = ('0' + (date.getMonth() + 1)).slice(-2);
                        var d = ('0' + date.getDate()).slice(-2);
                        var key = y + '-' + m + '-' + d;

                        if (enabledSet[key]) {
                            return [true, '', 'Available'];
                        }
                        return [false, '', 'Unavailable'];
                    },
                    onSelect: function(dateText) {
                        var daySlots = slotsMap[dateText] || [];
                        var html = '';

                        if (!daySlots.length) {
                            html = '<p class="ttb-timeslots__notice"><?php echo esc_js( __( 'No times available for this date.', 'ttb-woo-timeslot-bookings' ) ); ?></p>';
                        } else {
                            html = '<div class="ttb-timeslots__options">';
                            daySlots.forEach(function(slot, idx) {
                                var parts = slot.split('|'); // [start, end]
                                var label = parts[0].substring(0,5) + ' - ' + parts[1].substring(0,5);
                                var val = dateText + '|' + parts[0] + '|' + parts[1];
                                var id = 'ttb_slot_' + dateText.replace(/-/g, '') + '_' + idx;

                                html += '<label class="ttb-timeslots__option" for="' + id + '">';
                                html += '<input type="radio" id="' + id + '" name="ttb_time_choice" value="' + val + '"> ';
                                html += label + '</label>';
                            });
                            html += '</div>';
                        }

                        $timesWrap.html(html);
                        $valueInput.val('');

                        $timesWrap.find('input[type="radio"]').on('change', function() {
                            $valueInput.val($(this).val());
                        });
                    }
                });
            });
        </script>
        <?php
    }

    /**
     * Build calendar data:
     * - enabled_dates: dates that have at least one free slot
     * - slots_map: date => [ "start|end", "start|end" ]
     * Slots are shared globally across products.
     */
    protected function get_calendar_data() {
        $enabled_dates = array();
        $slots_map     = array();

        $today    = current_time( 'Y-m-d' );
        $start_ts = strtotime( $today );
        $end_ts   = strtotime( '+' . self::DAYS_AHEAD . ' days', $start_ts );

        $slots_config = $this->get_defined_slots();
        if ( empty( $slots_config ) ) {
            return array(
                'enabled_dates' => array(),
                'slots_map'     => array(),
            );
        }

        global $wpdb;
        $table = $this->get_table_name();

        for ( $ts = $start_ts; $ts <= $end_ts; $ts = strtotime( '+1 day', $ts ) ) {
            $weekday = date( 'N', $ts ); // 1 = Mon, 7 = Sun

            // Skip weekends
            if ( $weekday >= 6 ) {
                continue;
            }

            $date_sql = date( 'Y-m-d', $ts );

            // Skip blackout days
            if ( $this->is_blackout_date( $date_sql ) ) {
                continue;
            }

            // Fetch any booked/blocked starts for this date (any product)
            $booked_starts_raw = $wpdb->get_col(
                $wpdb->prepare(
                    "SELECT slot_start FROM {$table} WHERE slot_date = %s AND status IN ('booked','blocked')",
                    $date_sql
                )
            );

            // Normalise times from DB to HH:MM so they match the slot config
            $booked_short = array();
            if ( ! empty( $booked_starts_raw ) ) {
                foreach ( (array) $booked_starts_raw as $t ) {
                    $booked_short[] = substr( $t, 0, 5 );
                }
            }

            $available_slots = array();

            foreach ( $slots_config as $slot ) {
                $start = $slot['start'];
                $end   = $slot['end'];

                if ( ! in_array( $start, $booked_short, true ) ) {
                    $available_slots[] = $start . '|' . $end;
                }
            }

            if ( ! empty( $available_slots ) ) {
                $enabled_dates[]        = $date_sql;
                $slots_map[ $date_sql ] = $available_slots;
            }
        }

        return array(
            'enabled_dates' => $enabled_dates,
            'slots_map'     => $slots_map,
        );
    }

    /**
     * Check if a date is blacked out.
     */
    protected function is_blackout_date( $date_sql ) {
        $blackouts = apply_filters( 'ttb_timeslot_blackout_dates', self::BLACKOUT_DATES );
        return in_array( $date_sql, $blackouts, true );
    }

    /**
     * Validate chosen timeslot when adding to cart.
     */
    public function validate_timeslot_on_add_to_cart( $passed, $product_id, $quantity ) {
        if ( empty( $_POST['ttb_timeslot'] ) ) {
            wc_add_notice( __( 'Please select a date and time slot.', 'ttb-woo-timeslot-bookings' ), 'error' );
            return false;
        }

        $timeslot_raw = sanitize_text_field( wp_unslash( $_POST['ttb_timeslot'] ) );
        $parts        = explode( '|', $timeslot_raw );

        if ( count( $parts ) !== 3 ) {
            wc_add_notice( __( 'Invalid timeslot selection.', 'ttb-woo-timeslot-bookings' ), 'error' );
            return false;
        }

        list( $date, $start, $end ) = $parts;

        $start_short = substr( $start, 0, 5 );

        // Check slot still free globally
        if ( ! $this->is_slot_available( $date, $start_short ) ) {
            wc_add_notice( __( 'That timeslot has just been taken. Please choose another.', 'ttb-woo-timeslot-bookings' ), 'error' );
            return false;
        }

        // No blackout days
        if ( $this->is_blackout_date( $date ) ) {
            wc_add_notice( __( 'This date is not available for bookings. Please choose another.', 'ttb-woo-timeslot-bookings' ), 'error' );
            return false;
        }

        // No weekends
        $weekday = date( 'N', strtotime( $date ) );
        if ( $weekday >= 6 ) {
            wc_add_notice( __( 'Weekends are not available for bookings.', 'ttb-woo-timeslot-bookings' ), 'error' );
            return false;
        }

        return $passed;
    }

    /**
     * Check if a given slot is still available globally.
     * $start is expected as HH:MM.
     */
    protected function is_slot_available( $date, $start ) {
        global $wpdb;
        $table = $this->get_table_name();

        $found = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT id FROM {$table} WHERE slot_date = %s AND slot_start = %s AND status IN ('booked','blocked')",
                $date,
                $start
            )
        );

        return empty( $found );
    }

    /**
     * Attach timeslot data to cart item.
     */
    public function add_timeslot_to_cart_item( $cart_item_data, $product_id, $variation_id ) {
        if ( empty( $_POST['ttb_timeslot'] ) ) {
            return $cart_item_data;
        }

        $timeslot_raw = sanitize_text_field( wp_unslash( $_POST['ttb_timeslot'] ) );
        $parts        = explode( '|', $timeslot_raw );

        if ( count( $parts ) !== 3 ) {
            return $cart_item_data;
        }

        list( $date, $start, $end ) = $parts;

        $cart_item_data['ttb_timeslot'] = array(
            'date'  => $date,
            'start' => $start,
            'end'   => $end,
        );

        // Give it a unique key so identical bookings do not get merged
        $cart_item_data['unique_key'] = md5( microtime() . rand() );

        return $cart_item_data;
    }

    /**
     * Show timeslot info in cart and checkout.
     */
    public function display_timeslot_in_cart( $item_data, $cart_item ) {
        if ( empty( $cart_item['ttb_timeslot'] ) ) {
            return $item_data;
        }

        $slot       = $cart_item['ttb_timeslot'];
        $date_label = date_i18n( 'D j M Y', strtotime( $slot['date'] ) );

        $start_label = substr( $slot['start'], 0, 5 );
        $end_label   = substr( $slot['end'], 0, 5 );

        $item_data[] = array(
            'name'  => __( 'Timeslot', 'ttb-woo-timeslot-bookings' ),
            'value' => sprintf( '%s %s - %s', $date_label, $start_label, $end_label ),
        );

        return $item_data;
    }

    /**
     * Persist timeslot data into order line items.
     */
    public function add_timeslot_to_order_items( $item, $cart_item_key, $values, $order ) {
        if ( empty( $values['ttb_timeslot'] ) ) {
            return;
        }

        $slot = $values['ttb_timeslot'];

        $item->add_meta_data( '_ttb_timeslot_date', $slot['date'], true );
        $item->add_meta_data( '_ttb_timeslot_start', $slot['start'], true );
        $item->add_meta_data( '_ttb_timeslot_end', $slot['end'], true );
    }

    /**
     * When order is paid, lock all timeslots from the order.
     * One booking per slot across all products.
     */
    public function lock_timeslots_for_order( $order_id ) {
        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return;
        }

        foreach ( $order->get_items() as $item_id => $item ) {
            $product_id = $item->get_product_id();
            $date       = $item->get_meta( '_ttb_timeslot_date', true );
            $start      = $item->get_meta( '_ttb_timeslot_start', true );
            $end        = $item->get_meta( '_ttb_timeslot_end', true );

            if ( ! $product_id || ! $date || ! $start || ! $end ) {
                continue;
            }

            $this->book_slot( $product_id, $date, $start, $end, $order_id );
        }
    }

    /**
     * Insert a booking record into the custom table.
     */
    protected function book_slot( $product_id, $date, $start, $end, $order_id ) {
        $start_short = substr( $start, 0, 5 );
        $end_short   = substr( $end, 0, 5 );

        // If slot already booked or blocked by any product, do nothing
        if ( ! $this->is_slot_available( $date, $start_short ) ) {
            return;
        }

        global $wpdb;
        $table = $this->get_table_name();

        $wpdb->insert(
            $table,
            array(
                'product_id' => $product_id,
                'slot_date'  => $date,
                'slot_start' => $start_short,
                'slot_end'   => $end_short,
                'order_id'   => $order_id,
                'status'     => 'booked',
            ),
            array(
                '%d',
                '%s',
                '%s',
                '%s',
                '%d',
                '%s',
            )
        );
    }
}

new TTB_Woo_Timeslot_Bookings();
